<?php
// save.php
header('Content-Type: application/json');

// Check if it's a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['status' => 'error', 'message' => 'Method not allowed']);
    exit;
}

// Get the raw POST data
$input = file_get_contents('php://input');
if (empty($input)) {
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => 'No data received']);
    exit;
}

$data = json_decode($input, true);
if ($data === null) {
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => 'Invalid JSON data']);
    exit;
}

// Required files that must be saved
$files = [
    'antibot' => 'antibot.json',
    'blocked_ips' => 'blocked_ips.json',
    'blocked_asns' => 'blocked_asns.json',
    'lesta_conf' => 'lesta_conf.json'
];

$success = true;
$errors = [];
$savedFiles = [];

// Save required files
foreach ($files as $key => $filename) {
    if (!isset($data[$key])) {
        // Set default empty values for missing data
        if (in_array($key, ['antibot', 'lesta_conf'])) {
            $data[$key] = [];
        } elseif (in_array($key, ['blocked_ips', 'blocked_asns'])) {
            $data[$key] = [];
        } else {
            $errors[] = "Missing data for: $key";
            $success = false;
            continue;
        }
    }
    
    $content = $data[$key];
    
    // Pretty print the JSON
    $jsonContent = json_encode($content, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    
    if ($jsonContent === false) {
        $errors[] = "Failed to encode JSON for $filename";
        $success = false;
        continue;
    }
    
    // Write to file
    $result = file_put_contents($filename, $jsonContent);
    
    if ($result === false) {
        $errors[] = "Failed to write to $filename (check permissions)";
        $success = false;
    } else {
        $savedFiles[] = $filename;
        chmod($filename, 0644);
    }
}

// Save optional asn_cache if present
if (isset($data['asn_cache'])) {
    $jsonContent = json_encode($data['asn_cache'], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    if ($jsonContent !== false && file_put_contents('asn_cache.json', $jsonContent) !== false) {
        $savedFiles[] = 'asn_cache.json';
        chmod('asn_cache.json', 0644);
    }
}

if ($success) {
    echo json_encode([
        'status' => 'success', 
        'message' => 'Files saved successfully',
        'saved_files' => $savedFiles
    ]);
} else {
    http_response_code(500);
    echo json_encode([
        'status' => 'error', 
        'message' => 'Failed to save files',
        'errors' => $errors,
        'saved_files' => $savedFiles
    ]);
}
?>