<?php
/**
 * AntiBot Endpoint - Simple IP Blocker
 * Blocks bots, VPNs, datacenters, and suspicious IPs
 * Auto-creates all configuration and log files
 * Skips scanning for localhost IPs
 */

declare(strict_types=1);

// =============================
// Auto-create Directory Structure
// =============================
$ASSETS_DIR = __DIR__ . '/assets';
$CONFIG_DIR = $ASSETS_DIR . '/config';
$LOGS_DIR   = $ASSETS_DIR . '/logs';

// Create directories if they don't exist
foreach ([$ASSETS_DIR, $CONFIG_DIR, $LOGS_DIR] as $dir) {
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
}

// =============================
// File Paths
// =============================
$CONFIG_FILE        = $CONFIG_DIR . '/antibot.json';
$LOG_FILE           = $LOGS_DIR . '/visitor_logs.jsonl';
$BLOCKED_IPS_FILE   = $CONFIG_DIR . '/blocked_ips.json';
$BLOCKED_ASNS_FILE  = $CONFIG_DIR . '/blocked_asns.json';

// =============================
// Default Configuration
// =============================
$defaultConfig = [
    'target_bot'         => 'https://www.google.com',
    'block_datacenter'   => true,
    'block_recursive'    => true,
    'block_spamhaus'     => true,
    'block_blackbox'     => true,
    'enable_scamalytics' => true,
    'ipinfo_token'       => 'f9d6a179c3b8eb', // Free tier token
    'auto_update_lists'  => true,
    'log_all_visits'     => true,
    'stealth_mode'       => true,
    'skip_localhost'     => true, // New option: skip scanning for localhost
    'datacenter_keywords' => [
        'amazon', 'aws', 'google', 'gcp', 'microsoft', 'azure',
        'digitalocean', 'linode', 'vultr', 'ovh', 'leaseweb',
        'alibaba', 'tencent', 'cloudflare', 'contabo', 'hetzner',
        'rackspace', 'oracle cloud', 'ibm cloud', 'backblaze',
        'datacenter', 'server', 'hosting', 'colo', 'cloud'
    ]
];

// Default blocked ASNs (common hosting/VPN providers)
$defaultBlockedASNs = [
    'AS16509',  // Amazon AWS
    'AS15169',  // Google
    'AS8075',   // Microsoft
    'AS14618',  // Amazon
    'AS14061',  // DigitalOcean
    'AS20473',  // Choopa (Vultr)
    'AS16276',  // OVH
    'AS13335',  // Cloudflare
    'AS24940',  // Hetzner
    'AS12876',  // Online SAS
    'AS3257',   // GTT
    'AS3549',   // Level3
    'AS7018',   // AT&T
    'AS7922',   // Comcast
    'AS20940',  // Akamai
    'AS32934',  // Facebook
    'AS36040',  // TikTok
    'AS38365',  // 12306
    'AS26415',  // GoDaddy
    'AS46606'   // Unified Layer
];

// =============================
// Helper Functions
// =============================
function ensure_file_exists(string $path, string $defaultContent = ''): void {
    if (!file_exists($path)) {
        $dir = dirname($path);
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }
        file_put_contents($path, $defaultContent, LOCK_EX);
    }
}

function load_config(string $path, array $defaults): array {
    ensure_file_exists($path, json_encode($defaults, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES));
    
    $raw = file_get_contents($path);
    $cfg = json_decode($raw ?: '', true);
    
    if (!is_array($cfg)) {
        $cfg = $defaults;
        file_put_contents($path, json_encode($cfg, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES), LOCK_EX);
    }
    
    // Merge with defaults, preserving user settings
    foreach ($defaults as $key => $value) {
        if (!array_key_exists($key, $cfg)) {
            $cfg[$key] = $value;
        }
    }
    
    return $cfg;
}

function load_json_array(string $path, array $default = []): array {
    ensure_file_exists($path, json_encode($default, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES));
    
    $raw = file_get_contents($path);
    $arr = json_decode($raw ?: '', true);
    
    return is_array($arr) ? $arr : $default;
}

function save_json_array(string $path, array $arr): void {
    $dir = dirname($path);
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
    file_put_contents($path, json_encode($arr, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES), LOCK_EX);
}

function is_ip_blocked(string $ip, array $blocked_ips): bool {
    foreach ($blocked_ips as $blocked_ip) {
        if (strpos($blocked_ip, '/') !== false) {
            if (cidr_match($ip, $blocked_ip)) return true;
        } else {
            if ($ip === $blocked_ip) return true;
        }
    }
    return false;
}

function cidr_match(string $ip, string $cidr): bool {
    if (strpos($ip, ':') !== false || strpos($cidr, ':') !== false) return false;
    [$subnet, $mask] = explode('/', $cidr) + [null, null];
    $mask = (int)$mask;
    if ($mask < 0 || $mask > 32) return false;
    $ip_long = ip2long($ip);
    $subnet_long = ip2long($subnet);
    if ($ip_long === false || $subnet_long === false) return false;
    $mask_long = -1 << (32 - $mask);
    $subnet_long &= $mask_long;
    return ($ip_long & $mask_long) === $subnet_long;
}

function check_spamhaus(string $ip, int $timeout = 4): array {
    $url = "https://spamhaus.org/api/v1/sia-proxy/api/intel/v1/byobject/cidr/ALL/listings/live/" . rawurlencode($ip);
    
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => $timeout,
        CURLOPT_CONNECTTIMEOUT => $timeout,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/100.0.0.0 Safari/537.36',
        CURLOPT_HTTPHEADER => [
            'Accept: application/json',
            'Referer: https://www.spamhaus.org/lookup/'
        ]
    ]);
    
    $resp = curl_exec($ch);
    $err  = curl_error($ch);
    curl_close($ch);

    if ($resp === false || !empty($err)) {
        return ['listed' => false, 'lists' => [], 'list_names' => []];
    }
    
    $json = json_decode($resp, true);
    
    if (!is_array($json)) {
        return ['listed' => false, 'lists' => [], 'list_names' => []];
    }

    if (isset($json['code']) && $json['code'] === 404) {
        return ['listed' => false, 'lists' => [], 'list_names' => []];
    }
    
    if (!empty($json['results']) && is_array($json['results'])) {
        $list_names = [];
        foreach ($json['results'] as $entry) {
            if (isset($entry['dataset']) && !in_array($entry['dataset'], $list_names, true)) {
                $list_names[] = $entry['dataset'];
            }
        }
        if (empty($list_names)) {
            $list_names[] = 'SPAMHAUS';
        }

        return [
            'listed' => true,
            'lists' => $json['results'],
            'list_names' => $list_names
        ];
    }
    
    return ['listed' => false, 'lists' => [], 'list_names' => []];
}

function check_blackbox(string $ip, int $timeout = 3): bool {
    $url = "https://blackbox.ipinfo.app/lookup/" . rawurlencode($ip);
    $ch = curl_init($url);
    
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => $timeout,
        CURLOPT_CONNECTTIMEOUT => $timeout,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_USERAGENT => 'antibot-script',
    ]);
    
    $resp = curl_exec($ch);
    $err  = curl_error($ch);
    curl_close($ch);

    if ($resp === false || !empty($err)) {
        return false;
    }
    
    return (trim($resp) === 'Y');
}

function is_asn_blocked(string $asn, array $blocked_asns): bool {
    return in_array(strtoupper($asn), array_map('strtoupper', $blocked_asns), true);
}

function count_ip_in_logs(string $logFile, string $ip): int {
    if (!file_exists($logFile)) {
        return 0;
    }

    $count = 0;
    $fh = @fopen($logFile, 'r');
    if (!$fh) {
        return 0;
    }

    $search_string = '"ip":"' . $ip . '"';

    while (($line = fgets($fh)) !== false) {
        if (strpos($line, $search_string) !== false) {
            $count++;
            if ($count >= 3) {
                break;
            }
        }
    }

    fclose($fh);
    return $count;
}

function check_scamalytics(string $ip, int $timeout = 5): bool {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, 'https://scamalytics.com/ip/' . $ip);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'GET');
    curl_setopt($ch, CURLOPT_TIMEOUT, $timeout);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/142.0.0.0 Safari/537.36 Edg/142.0.0.0');
    
    $response = curl_exec($ch);
    $err = curl_error($ch);
    curl_close($ch);
    
    if ($response === false || $err) return false;
    return strpos($response, '<div class="risk yes">Yes</div>') !== false;
}

function ipinfo_lite(?string $token, string $ip, int $timeout=5): array {
    $empty = ['asn'=>'','as_name'=>'','country_code'=>'','country'=>'','privacy'=>[]];
    if (!$token) return $empty;
    $url = "https://api.ipinfo.io/lite/".rawurlencode($ip)."?token=".rawurlencode($token);
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => $timeout,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
    ]);
    $resp = curl_exec($ch);
    $err  = curl_error($ch);
    curl_close($ch);
    if ($resp && !$err) {
        $j = json_decode($resp, true);
        if (is_array($j)) {
            return [
                'asn' => strtoupper((string)($j['asn'] ?? '')),
                'as_name' => strtolower((string)($j['as_name'] ?? '')),
                'country_code' => strtoupper((string)($j['country_code'] ?? '')),
                'country' => $j['country'] ?? '',
                'privacy' => $j['privacy'] ?? [],
            ];
        }
    }
    return $empty;
}

function client_ip(): string {
    // Priority 1: GET parameter (for testing)
    if (isset($_GET['ip'])) {
        $ip = filter_var($_GET['ip'], FILTER_VALIDATE_IP);
        if ($ip !== false) return $ip;
    }
    
    // Priority 2: POST parameter
    if (isset($_POST['ip'])) {
        $ip = filter_var($_POST['ip'], FILTER_VALIDATE_IP);
        if ($ip !== false) return $ip;
    }
    
    // Priority 3: CloudFlare
    if (isset($_SERVER["HTTP_CF_CONNECTING_IP"])) {
        $ip = filter_var($_SERVER["HTTP_CF_CONNECTING_IP"], FILTER_VALIDATE_IP);
        if ($ip !== false) return $ip;
    }
    
    // Priority 4: Standard headers
    $headers = ['HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'HTTP_CLIENT_IP', 'REMOTE_ADDR'];
    foreach ($headers as $header) {
        if (!empty($_SERVER[$header])) {
            $ip_list = $_SERVER[$header];
            // Handle comma-separated lists (like X-Forwarded-For)
            if (strpos($ip_list, ',') !== false) {
                $ips = explode(',', $ip_list);
                $ip = trim($ips[0]);
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            } else {
                $ip = filter_var($ip_list, FILTER_VALIDATE_IP);
                if ($ip !== false) return $ip;
            }
        }
    }
    
    return '0.0.0.0';
}

function log_visit(string $path, array $row): void {
    ensure_file_exists($path, '');
    file_put_contents($path, json_encode($row, JSON_UNESCAPED_SLASHES).PHP_EOL, FILE_APPEND|LOCK_EX);
}

function block_response(string $message = 'Blocked'): void {
    if (!headers_sent()) {
        usleep(random_int(400000, 1500000));
        header("HTTP/1.1 200 OK");
        header("Content-Type: text/html; charset=UTF-8");
    }
    echo "<!-- $message -->";
    exit; // THIS EXITS ONLY FOR BLOCKED VISITORS
}

function is_localhost_ip(string $ip): bool {
    // Common localhost IPv4 addresses
    $localhost_ips_v4 = [
        '127.0.0.1',
        '127.0.0.0',
        '127.255.255.255',
        '0.0.0.0',
        'localhost'
    ];
    
    // Common localhost IPv6 addresses
    $localhost_ips_v6 = [
        '::1',
        '::',
        '0:0:0:0:0:0:0:1',
        '0:0:0:0:0:0:0:0'
    ];
    
    // Private IP ranges (RFC 1918, RFC 4193, RFC 5735)
    $private_ranges = [
        '10.0.0.0/8',
        '172.16.0.0/12',
        '192.168.0.0/16',
        'fc00::/7',  // IPv6 Unique Local Addresses
        'fe80::/10', // IPv6 Link-Local Addresses
        '169.254.0.0/16', // Link-local addresses
        '100.64.0.0/10', // Shared Address Space
    ];
    
    // Check exact matches
    if (in_array($ip, $localhost_ips_v4, true) || in_array($ip, $localhost_ips_v6, true)) {
        return true;
    }
    
    // Check private ranges for IPv4
    if (strpos($ip, ':') === false) { // IPv4
        foreach ($private_ranges as $range) {
            if (strpos($range, ':') === false && cidr_match($ip, $range)) {
                return true;
            }
        }
    }
    
    // Simple check for common patterns
    if (str_starts_with($ip, '192.168.') ||
        str_starts_with($ip, '10.') ||
        (str_starts_with($ip, '172.') && preg_match('/^172\.(1[6-9]|2[0-9]|3[0-1])\./', $ip)) ||
        str_starts_with($ip, 'fe80:') ||
        str_starts_with($ip, 'fc') ||
        str_starts_with($ip, 'fd')) {
        return true;
    }
    
    return false;
}

function update_block_lists(array &$blocked_asns): void {
    global $CONFIG_DIR, $BLOCKED_ASNS_FILE;
    
    // Update blocked ASNs list periodically (once per day)
    $asn_cache_file = $CONFIG_DIR . '/asn_cache.json';
    $last_update = 0;
    
    if (file_exists($asn_cache_file)) {
        $cache = json_decode(file_get_contents($asn_cache_file), true);
        $last_update = $cache['last_update'] ?? 0;
    }
    
    // Update if more than 24 hours old
    if (time() - $last_update > 86400) {
        // Add new ASNs to the list
        $new_asns = [
            'AS6939',   // Hurricane Electric
            'AS1299',   // Telia
            'AS3320',   // Deutsche Telekom
            'AS3215',   // Orange
            'AS6830',   // Liberty Global
            'AS2856',   // BT
            'AS1273',   // Vodafone
            'AS701',    // Verizon
            'AS6130',   // Comcast
            'AS20001',  // Verizon Business
        ];
        
        foreach ($new_asns as $asn) {
            if (!in_array($asn, $blocked_asns)) {
                $blocked_asns[] = $asn;
            }
        }
        
        // Save updated list
        save_json_array($BLOCKED_ASNS_FILE, $blocked_asns);
        
        // Update cache
        file_put_contents($asn_cache_file, json_encode([
            'last_update' => time(),
            'total_asns' => count($blocked_asns)
        ], JSON_PRETTY_PRINT), LOCK_EX);
    }
}

// =============================
// Initialize System
// =============================

// Create initial log file if it doesn't exist
ensure_file_exists($LOG_FILE, '');

// Load configuration and lists
$config       = load_config($CONFIG_FILE, $defaultConfig);
$blocked_ips  = load_json_array($BLOCKED_IPS_FILE, []);
$blocked_asns = load_json_array($BLOCKED_ASNS_FILE, $defaultBlockedASNs);

// Auto-update block lists if enabled
if ($config['auto_update_lists'] ?? true) {
    update_block_lists($blocked_asns);
}

// =============================
// Get visitor information
// =============================
$ip   = client_ip();
$ua   = $_SERVER['HTTP_USER_AGENT'] ?? '';
$path = $_SERVER['REQUEST_URI'] ?? '/';

// =============================
// Skip scanning for localhost IPs
// =============================
if (($config['skip_localhost'] ?? true) && is_localhost_ip($ip)) {
    // Log the localhost visit but don't block
    log_visit($LOG_FILE, [
        'time'=>gmdate('Y-m-d H:i:s'),'ip'=>$ip,'ua'=>$ua,'verdict'=>'localhost','path'=>$path,
        'asn'=>'','as_name'=>'','country_code'=>'','country'=>'','blocked_reason'=>'',
        'challenge_pass'=>true,'fp_verified'=>true,'action'=>'allowed',
        'scamalytics'=>false,
        'spamhaus' => false,
        'blackbox' => false,
    ]);
    
    // Allow access immediately - just output a comment
    echo "<!-- localhost access granted for $ip -->\n";
    // DO NOT EXIT - let index.php continue
    return; // This returns from the included file, not exit the whole script
}

// =============================
// Main Processing (only for non-localhost IPs)
// =============================

// Recursive IP Block (Log Count)
if (!empty($config['block_recursive'])) {
    $hit_count = count_ip_in_logs($LOG_FILE, $ip);
    
    if ($hit_count >= 3) {
        if (!in_array($ip, $blocked_ips, true)) {
            $blocked_ips[] = $ip;
            save_json_array($BLOCKED_IPS_FILE, $blocked_ips);
        }
        log_visit($LOG_FILE, [
            'time'=>gmdate('Y-m-d H:i:s'),'ip'=>$ip,'ua'=>$ua,'verdict'=>'bot','path'=>$path,
            'asn'=>'','as_name'=>'','country_code'=>'','country'=>'','blocked_reason'=>'recursive_log_count',
            'challenge_pass'=>false,'fp_verified'=>false,'action'=>'stealth',
            'scamalytics'=>false,
            'spamhaus' => false,
            'blackbox' => false,
        ]);
        block_response('Blocked: Too many visits');
        // This exits the script for blocked visitors
    }
}

// Spamhaus Check
if (!empty($config['block_spamhaus'])) {
    $spamhaus = check_spamhaus($ip);
    if ($spamhaus['listed']) {
        if (!in_array($ip, $blocked_ips, true)) {
            $blocked_ips[] = $ip;
            save_json_array($BLOCKED_IPS_FILE, $blocked_ips);
        }
        log_visit($LOG_FILE, [
            'time'=>gmdate('Y-m-d H:i:s'),'ip'=>$ip,'ua'=>$ua,'verdict'=>'bot','path'=>$path,
            'asn'=>'','as_name'=>'','country_code'=>'','country'=>'','blocked_reason'=>'spamhaus_'.implode('_', $spamhaus['list_names']),
            'challenge_pass'=>false,'fp_verified'=>false,'action'=>'stealth',
            'scamalytics'=>false,'spamhaus'=>true,
            'blackbox' => false,
        ]);
        block_response('Blocked: Listed in Spamhaus');
        // This exits the script for blocked visitors
    }
}

// Blocked IPs Check
if (is_ip_blocked($ip, $blocked_ips)) {
    log_visit($LOG_FILE, [
        'time'=>gmdate('Y-m-d H:i:s'),'ip'=>$ip,'ua'=>$ua,'verdict'=>'bot','path'=>$path,
        'asn'=>'','as_name'=>'','country_code'=>'','country'=>'','blocked_reason'=>'ip_blocked',
        'challenge_pass'=>false,'fp_verified'=>false,'action'=>'redirect_bot',
        'scamalytics'=>false,
        'spamhaus' => false,
        'blackbox' => false,
    ]);
    header('Location: '.$config['target_bot'], true, 302);
    exit; // This exits the script for blocked visitors
}

// Scamalytics Check
$scam = false;
if (!empty($config['enable_scamalytics']) && check_scamalytics($ip, 5)) {
    $scam = true;
    if (!in_array($ip, $blocked_ips, true)) {
        $blocked_ips[] = $ip;
        save_json_array($BLOCKED_IPS_FILE, $blocked_ips);
    }
    log_visit($LOG_FILE, [
        'time'=>gmdate('Y-m-d H:i:s'),'ip'=>$ip,'ua'=>$ua,'verdict'=>'bot','path'=>$path,
        'asn'=>'','as_name'=>'','country_code'=>'','country'=>'','blocked_reason'=>'scamalytics_bot',
        'challenge_pass'=>false,'fp_verified'=>false,'action'=>'stealth',
        'scamalytics'=>true,
        'spamhaus' => false,
        'blackbox' => false,
    ]);
    block_response('Blocked: High fraud risk');
    // This exits the script for blocked visitors
}

// IPinfo for ASN and datacenter detection
$lite = ipinfo_lite($config['ipinfo_token'] ?? '', $ip);
$client_asn = strtoupper((string)($lite['asn'] ?? ''));
$client_as_name = strtolower((string)($lite['as_name'] ?? ''));

// Datacenter Block
if (!empty($config['block_datacenter'])) {
    $dc_keywords = $config['datacenter_keywords'] ?? $defaultConfig['datacenter_keywords'];
    foreach ($dc_keywords as $dc) {
        if ($dc !== '' && strpos($client_as_name, $dc) !== false) {
            if (!in_array($ip, $blocked_ips, true)) {
                $blocked_ips[] = $ip;
                save_json_array($BLOCKED_IPS_FILE, $blocked_ips);
            }
            log_visit($LOG_FILE, [
                'time'=>gmdate('Y-m-d H:i:s'),'ip'=>$ip,'ua'=>$ua,'verdict'=>'bot','path'=>$path,
                'asn'=>$client_asn,'as_name'=>$client_as_name,'country_code'=>$lite['country_code']??'','country'=>$lite['country']??'',
                'blocked_reason'=>'datacenter_auto','challenge_pass'=>false,'fp_verified'=>false,'action'=>'stealth',
                'scamalytics'=>$scam,
                'spamhaus' => false,
                'blackbox' => false,
            ]);
            block_response('Blocked: Datacenter IP');
            // This exits the script for blocked visitors
        }
    }
}

// ASN Block
if ($client_asn && is_asn_blocked($client_asn, $blocked_asns)) {
    if (!in_array($ip, $blocked_ips, true)) {
        $blocked_ips[] = $ip;
        save_json_array($BLOCKED_IPS_FILE, $blocked_ips);
    }
    log_visit($LOG_FILE, [
        'time'=>gmdate('Y-m-d H:i:s'),'ip'=>$ip,'ua'=>$ua,'verdict'=>'bot','path'=>$path,
        'asn'=>$client_asn,'as_name'=>$client_as_name,'country_code'=>$lite['country_code']??'','country'=>$lite['country']??'',
        'blocked_reason'=>'asn_block','challenge_pass'=>false,'fp_verified'=>false,'action'=>'stealth',
        'scamalytics'=>$scam,
        'spamhaus' => false,
        'blackbox' => false,
    ]);
    block_response('Blocked: Blocked ASN');
    // This exits the script for blocked visitors
}

// Blackbox Check
if (!empty($config['block_blackbox'])) {
    if (check_blackbox($ip)) {
        if (!in_array($ip, $blocked_ips, true)) {
            $blocked_ips[] = $ip;
            save_json_array($BLOCKED_IPS_FILE, $blocked_ips);
        }
        log_visit($LOG_FILE, [
            'time'=>gmdate('Y-m-d H:i:s'),'ip'=>$ip,'ua'=>$ua,'verdict'=>'bot','path'=>$path,
            'asn'=>$client_asn,'as_name'=>$client_as_name,'country_code'=>$lite['country_code']??'','country'=>$lite['country']??'',
            'blocked_reason'=>'blackbox_vpn_proxy',
            'challenge_pass'=>false,'fp_verified'=>false,'action'=>'stealth',
            'scamalytics'=>$scam,
            'spamhaus'=>false,
            'blackbox'=>true,
        ]);
        block_response('Blocked: VPN/Proxy detected');
        // This exits the script for blocked visitors
    }
}

// =============================
// If all checks pass - ALLOW THE VISITOR
// =============================
log_visit($LOG_FILE, [
    'time'=>gmdate('Y-m-d H:i:s'),'ip'=>$ip,'ua'=>$ua,'verdict'=>'human','path'=>$path,
    'asn'=>$client_asn,'as_name'=>$client_as_name,'country_code'=>$lite['country_code']??'','country'=>$lite['country']??'',
    'blocked_reason'=>'','challenge_pass'=>true,'fp_verified'=>true,'action'=>'allowed',
    'scamalytics'=>$scam,
    'spamhaus'=>false,
    'blackbox'=>false,
]);

// Show success response but DO NOT EXIT
if ($config['stealth_mode'] ?? true) {
    // Just output a comment and let the script continue
    echo "<!-- antibot: access granted for $ip -->\n";
    // IMPORTANT: No exit() here - script continues to index.php
} else {
    // If not in stealth mode and someone wants to see the response
    if (isset($_SERVER['HTTP_ACCEPT']) && strpos($_SERVER['HTTP_ACCEPT'], 'application/json') !== false) {
        header('Content-Type: application/json');
        echo json_encode([
            'status' => 'allowed',
            'ip' => $ip,
            'asn' => $client_asn,
            'as_name' => $client_as_name,
            'country' => $lite['country'] ?? 'Unknown',
            'country_code' => $lite['country_code'] ?? '',
            'timestamp' => date('Y-m-d H:i:s'),
            'system' => 'antibot',
            'version' => '2.0'
        ], JSON_PRETTY_PRINT);
        exit; // Exit here because we sent JSON response
    } else {
        // Show HTML page but still exit
        echo "<!DOCTYPE html><html><head><title>Access Granted</title></head><body>";
        echo "<h1>✅ Access Granted</h1>";
        echo "<p>IP: " . htmlspecialchars($ip) . "</p>";
        if ($client_asn) echo "<p>ASN: " . htmlspecialchars($client_asn) . "</p>";
        if ($client_as_name) echo "<p>Network: " . htmlspecialchars($client_as_name) . "</p>";
        echo "</body></html>";
        exit; // Exit here because we sent full HTML response
    }
}

// If we reach here in stealth mode, the script continues to index.php
// No exit() called for allowed visitors in stealth mode